# -*- coding: utf-8 -*-

# videoplayer.py
# This file is part of qarte-5
#    
# Author: Vincent Vande Vyvre <vincent.vandevyvre@oqapy.eu>
# Copyright: 2018-2022 Vincent Vande Vyvre
# Licence: GPL3
# Home page: https://launchpad.net/qarte

# Qt multimedia
# https://doc.qt.io/qt-5/qmediaplayer.html
# https://doc.qt.io/qt-5/qgraphicsvideoitem.html
# https://doc.qt.io/qt-5/qvideowidget.html

from PyQt5.QtCore import QUrl
from PyQt5.QtMultimedia import QMediaPlayer, QMediaContent
from PyQt5.QtMultimediaWidgets import QGraphicsVideoItem, QVideoWidget

class VideoPlayer(QMediaPlayer):
    def __init__(self, driver, widget):
        super().__init__()
        self.driver = driver
        self.widget = widget
        self.setVideoOutput(widget)
        self.url = None
        self.auto_sound_level = True
        self.is_running = False
        self.is_on_pause = False
        self.volumeChanged.connect(self.on_volume_changed)
        self.stateChanged.connect(self.on_state_changed)

    def set_media(self, media):
        if media.startswith("http"):
            self.setMedia(QMediaContent(QUrl(media)))
        else:
            self.setMedia(QMediaContent(QUrl.fromLocalFile(media)))

    def start(self):
        self.is_running = True
        self.is_on_pause = False
        self.play()

    def set_on_pause(self):
        self.is_running = False
        self.is_on_pause = True
        self.pause()

    def finish(self):
        self.is_running = False
        self.is_on_pause = False
        self.stop()

    def set_sound_level(self, level):
        self.auto_sound_level = False
        self.setVolume(level)

    def on_volume_changed(self, value):
        if self.auto_sound_level:
            self.driver.page.update_volume_slider(value)
        self.auto_sound_level = True

    def on_state_changed(self, state):
        if not state:
            self.driver.stop_preview()



