/***************************************************************************
 *            gbemol-amazon.c
 *
 *  Tue Oct 30 00:01:04 2007
 *  Copyright  2007  felipe
 *  <felipe.weckx@gmail.com>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>

#include "gbemol-mpd.h"
#include "gbemol-amazon.h"
#include "gbemol-net.h"

static gchar* gbemol_amazon_create_url(GbemolMpdSong* song)
{
	GString *string = g_string_new(AWS_URL);
	gchar *url, *escaped;
		
	if (song->album)
	{
		escaped = gbemol_net_escape(song->album);
		g_string_append(string, AWS_SEARCH_TITLE);
		g_string_append(string, escaped);
	}
	
	if (song->artist)
	{
		escaped = gbemol_net_escape(song->artist);
		g_string_append(string, AWS_SEARCH_ARTIST);
		g_string_append(string, escaped);
	}
	
	url = string->str;
	g_string_free (string, FALSE);
	
	return url;
}

static xmlDoc* gbemol_amazon_parse_file(gchar *file_path)
{
	xmlDoc* doc;
	doc = xmlParseFile(file_path);

	return doc;
}

static xmlNodePtr gbemol_amazon_validate_doc(xmlDoc *doc)
{
	xmlNodePtr node;

	node = xmlDocGetRootElement (doc);
	if (node == NULL || xmlStrcmp (node->name, AWS_XML_ROOT)) {
		return NULL;
	}

	node = node->children;
	if (node == NULL || xmlStrcmp (node->name, AWS_XML_OREQ)) {
		return NULL;
	}

	node = node->next;
	if (node == NULL || xmlStrcmp (node->name, AWS_XML_ITEMS)) {
		return NULL;
	}

	return node->children;
}

static gchar* gbemol_amazon_get_childvalue (xmlNodePtr parent, xmlChar * childname)
{
	xmlNodePtr element;

	element = parent->children;
	while (element != NULL) {
		if (!xmlStrcmp (element->name, childname) && element->children != NULL) {
			return (gchar *) xmlStrdup (element->children->content);
		}
		element = element->next;
	}
	return NULL;
}

static gchar* gbemol_amazon_get_item_cover (xmlNodePtr element)
{
	while (element != NULL) {
		if (!xmlStrcmp (element->name, AWS_XML_ITEM_SMALLIMG)) {
			return gbemol_amazon_get_childvalue (element, AWS_XML_ITEM_IMG_URL);			
		} else if (!xmlStrcmp (element->name, AWS_XML_ITEM_MEDIUMIMG)) {
			return gbemol_amazon_get_childvalue (element, AWS_XML_ITEM_IMG_URL);
		} else if (!xmlStrcmp (element->name, AWS_XML_ITEM_LARGEIMG)) {
			return gbemol_amazon_get_childvalue (element, AWS_XML_ITEM_IMG_URL);
		} 				
		element = element->next;
	}

	return NULL;
}

static void gbemol_amazon_fetch_results(xmlNodePtr node, GSList **list)
{
	xmlNodePtr element;
	gchar *result;
	
	while (node != NULL) {
		if (!xmlStrcmp (node->name, AWS_XML_ITEM)) {
			element = node->children;
			if ((result = gbemol_amazon_get_item_cover(element)) != NULL) {
				*list = g_slist_prepend (*list, result);
			}
		}
		node = node->next;
	}
}


GSList* gbemol_amazon_search_cover(GbemolMpdSong* song)
{
	int fp;
	gchar *tmp_filename = NULL;
	xmlDoc* doc;
	xmlNodePtr node;
	GSList *results = NULL;
	gchar *url = gbemol_amazon_create_url(song);
	
	/* Try to create temporary file */
	if ((fp = g_file_open_tmp(NULL, &tmp_filename, NULL)) == -1)
	{
		g_free(url);
		g_free(tmp_filename);
		return NULL;
	}
	
	/* Try to download the url */
	if (!gbemol_net_download_file(url, tmp_filename, NULL))
	{		
		g_free(url);
		g_free(tmp_filename);
		return NULL;
	}
	g_free(url);
	
	doc = gbemol_amazon_parse_file(tmp_filename);
	if (doc == NULL)
	{		
		g_free(tmp_filename);
		return NULL;
	}
	close(fp);
	unlink(tmp_filename);
	g_free(tmp_filename);
	
	node = gbemol_amazon_validate_doc(doc);
	if (node == NULL)
	{
		return NULL;
	}
	
	gbemol_amazon_fetch_results(node, &results);
	xmlFreeDoc(doc);
	xmlCleanupParser();
	results = g_slist_reverse (results);

	return results;
}
 
