/*
 * junixsocket
 *
 * Copyright 2009-2022 Christian Kohlschütter
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.newsclub.net.unix.rmi;

import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInput;

/**
 * A specialized subclass of {@link RemoteFileDescriptorBase}, specifically for
 * {@link FileOutputStream}s.
 * 
 * @author Christian Kohlschütter
 */
public final class RemoteFileOutput extends RemoteFileDescriptorBase<FileOutputStream> {
  private static final long serialVersionUID = 1L;

  /**
   * Creates an uninitialized instance; used for externalization.
   * 
   * @see #readExternal(ObjectInput)
   */
  public RemoteFileOutput() {
    super();
  }

  /**
   * Creates a new {@link RemoteFileOutput} instance, encapsulating a {@link FileOutputStream} so
   * that it can be shared with other processes via RMI.
   * 
   * @param socketFactory The socket factory.
   * @param fout The {@link FileOutputStream}.
   * @throws IOException if the operation fails.
   */
  public RemoteFileOutput(AFUNIXRMISocketFactory socketFactory, FileOutputStream fout)
      throws IOException {
    super(socketFactory, fout, fout.getFD(), RemoteFileDescriptorBase.MAGIC_VALUE_MASK
        | RemoteFileDescriptorBase.BIT_WRITABLE);
  }

  /**
   * Returns a FileOutputStream for the given instance. This either is the actual instance provided
   * by the constructor or a new instance created from the file descriptor.
   * 
   * @return The FileOutputStream.
   * @throws IOException if the operation fails.
   */
  public FileOutputStream asFileOutputStream() throws IOException {
    if ((getMagicValue() & RemoteFileDescriptorBase.BIT_WRITABLE) == 0) {
      throw new IOException("FileDescriptor is not writable");
    }

    return resource.accumulateAndGet(null, (prev, x) -> {
      if (prev != null) {
        return prev;
      }
      FileDescriptor fd = getFileDescriptor();
      if (!fd.valid()) {
        throw new IllegalStateException("Invalid file descriptor");
      }
      return new FileOutputStream(fd) {
        @Override
        public synchronized void close() throws IOException {
          RemoteFileOutput.this.close();
          super.close();
        }
      };
    });
  }
}
