// Take a look at the license at the top of the repository in the LICENSE file.

use std::{mem, ptr};

use glib::{prelude::*, translate::*};

use crate::{
    ffi,
    utils::HasStreamLock,
    video_codec_state::{InNegotiation, Readable, VideoCodecState, VideoCodecStateContext},
    VideoCodecFrame, VideoEncoder,
};
mod sealed {
    pub trait Sealed {}
    impl<T: super::IsA<super::VideoEncoder>> Sealed for T {}
}

pub trait VideoEncoderExtManual: sealed::Sealed + IsA<VideoEncoder> + 'static {
    #[doc(alias = "gst_video_encoder_allocate_output_frame")]
    fn allocate_output_frame(
        &self,
        frame: &mut VideoCodecFrame,
        size: usize,
    ) -> Result<gst::FlowSuccess, gst::FlowError> {
        unsafe {
            try_from_glib(ffi::gst_video_encoder_allocate_output_frame(
                self.as_ref().to_glib_none().0,
                frame.to_glib_none().0,
                size,
            ))
        }
    }

    #[doc(alias = "get_frame")]
    #[doc(alias = "gst_video_encoder_get_frame")]
    fn frame(&self, frame_number: i32) -> Option<VideoCodecFrame> {
        let frame = unsafe {
            ffi::gst_video_encoder_get_frame(self.as_ref().to_glib_none().0, frame_number)
        };

        if frame.is_null() {
            None
        } else {
            unsafe { Some(VideoCodecFrame::new(frame, self.as_ref())) }
        }
    }

    #[doc(alias = "get_frames")]
    #[doc(alias = "gst_video_encoder_get_frames")]
    fn frames(&self) -> Vec<VideoCodecFrame> {
        unsafe {
            let frames = ffi::gst_video_encoder_get_frames(self.as_ref().to_glib_none().0);
            let mut iter: *const glib::ffi::GList = frames;
            let mut vec = Vec::new();

            while !iter.is_null() {
                let frame_ptr = Ptr::from((*iter).data);
                /* transfer ownership of the frame */
                let frame = VideoCodecFrame::new(frame_ptr, self.as_ref());
                vec.push(frame);
                iter = (*iter).next;
            }

            glib::ffi::g_list_free(frames);
            vec
        }
    }

    #[doc(alias = "get_oldest_frame")]
    #[doc(alias = "gst_video_encoder_get_oldest_frame")]
    fn oldest_frame(&self) -> Option<VideoCodecFrame> {
        let frame =
            unsafe { ffi::gst_video_encoder_get_oldest_frame(self.as_ref().to_glib_none().0) };

        if frame.is_null() {
            None
        } else {
            unsafe { Some(VideoCodecFrame::new(frame, self.as_ref())) }
        }
    }

    #[doc(alias = "get_allocator")]
    #[doc(alias = "gst_video_encoder_get_allocator")]
    fn allocator(&self) -> (Option<gst::Allocator>, gst::AllocationParams) {
        unsafe {
            let mut allocator = ptr::null_mut();
            let mut params = mem::MaybeUninit::uninit();
            ffi::gst_video_encoder_get_allocator(
                self.as_ref().to_glib_none().0,
                &mut allocator,
                params.as_mut_ptr(),
            );
            (from_glib_full(allocator), params.assume_init().into())
        }
    }

    #[cfg(feature = "v1_18")]
    #[cfg_attr(docsrs, doc(cfg(feature = "v1_18")))]
    #[doc(alias = "gst_video_encoder_finish_subframe")]
    fn finish_subframe(&self, frame: &VideoCodecFrame) -> Result<gst::FlowSuccess, gst::FlowError> {
        unsafe {
            try_from_glib(ffi::gst_video_encoder_finish_subframe(
                self.as_ref().to_glib_none().0,
                frame.to_glib_none().0,
            ))
        }
    }

    #[doc(alias = "get_latency")]
    #[doc(alias = "gst_video_encoder_get_latency")]
    fn latency(&self) -> (gst::ClockTime, Option<gst::ClockTime>) {
        let mut min_latency = gst::ffi::GST_CLOCK_TIME_NONE;
        let mut max_latency = gst::ffi::GST_CLOCK_TIME_NONE;

        unsafe {
            ffi::gst_video_encoder_get_latency(
                self.as_ref().to_glib_none().0,
                &mut min_latency,
                &mut max_latency,
            );

            (
                try_from_glib(min_latency).expect("undefined min_latency"),
                from_glib(max_latency),
            )
        }
    }

    #[doc(alias = "gst_video_encoder_set_latency")]
    fn set_latency(
        &self,
        min_latency: gst::ClockTime,
        max_latency: impl Into<Option<gst::ClockTime>>,
    ) {
        unsafe {
            ffi::gst_video_encoder_set_latency(
                self.as_ref().to_glib_none().0,
                min_latency.into_glib(),
                max_latency.into().into_glib(),
            );
        }
    }
    #[doc(alias = "get_output_state")]
    #[doc(alias = "gst_video_encoder_get_output_state")]
    fn output_state(&self) -> Option<VideoCodecState<'static, Readable>> {
        let state =
            unsafe { ffi::gst_video_encoder_get_output_state(self.as_ref().to_glib_none().0) };

        if state.is_null() {
            None
        } else {
            unsafe { Some(VideoCodecState::<Readable>::new(state)) }
        }
    }

    #[doc(alias = "gst_video_encoder_set_output_state")]
    fn set_output_state(
        &self,
        caps: gst::Caps,
        reference: Option<&VideoCodecState<Readable>>,
    ) -> Result<VideoCodecState<InNegotiation>, gst::FlowError> {
        let state = unsafe {
            let reference = match reference {
                Some(reference) => reference.as_mut_ptr(),
                None => ptr::null_mut(),
            };
            ffi::gst_video_encoder_set_output_state(
                self.as_ref().to_glib_none().0,
                caps.into_glib_ptr(),
                reference,
            )
        };

        if state.is_null() {
            Err(gst::FlowError::NotNegotiated)
        } else {
            unsafe { Ok(VideoCodecState::<InNegotiation>::new(state, self.as_ref())) }
        }
    }

    #[doc(alias = "gst_video_encoder_negotiate")]
    fn negotiate<'a>(
        &'a self,
        output_state: VideoCodecState<'a, InNegotiation<'a>>,
    ) -> Result<(), gst::FlowError> {
        // Consume output_state so user won't be able to modify it anymore
        let self_ptr = self.to_glib_none().0 as *const gst::ffi::GstElement;
        assert_eq!(output_state.context.element_as_ptr(), self_ptr);

        let ret = unsafe {
            from_glib(ffi::gst_video_encoder_negotiate(
                self.as_ref().to_glib_none().0,
            ))
        };
        if ret {
            Ok(())
        } else {
            Err(gst::FlowError::NotNegotiated)
        }
    }

    #[doc(alias = "gst_video_encoder_set_headers")]
    fn set_headers(&self, headers: impl IntoIterator<Item = gst::Buffer>) {
        unsafe {
            ffi::gst_video_encoder_set_headers(
                self.as_ref().to_glib_none().0,
                headers
                    .into_iter()
                    .collect::<glib::List<_>>()
                    .into_glib_ptr(),
            );
        }
    }

    fn sink_pad(&self) -> &gst::Pad {
        unsafe {
            let elt = &*(self.as_ptr() as *const ffi::GstVideoEncoder);
            &*(&elt.sinkpad as *const *mut gst::ffi::GstPad as *const gst::Pad)
        }
    }

    fn src_pad(&self) -> &gst::Pad {
        unsafe {
            let elt = &*(self.as_ptr() as *const ffi::GstVideoEncoder);
            &*(&elt.srcpad as *const *mut gst::ffi::GstPad as *const gst::Pad)
        }
    }

    fn input_segment(&self) -> gst::Segment {
        unsafe {
            let ptr: &ffi::GstVideoDecoder = &*(self.as_ptr() as *const _);
            glib::ffi::g_rec_mutex_lock(mut_override(&ptr.stream_lock));
            let segment = ptr.input_segment;
            glib::ffi::g_rec_mutex_unlock(mut_override(&ptr.stream_lock));
            from_glib_none(&segment as *const gst::ffi::GstSegment)
        }
    }

    fn output_segment(&self) -> gst::Segment {
        unsafe {
            let ptr: &ffi::GstVideoDecoder = &*(self.as_ptr() as *const _);
            glib::ffi::g_rec_mutex_lock(mut_override(&ptr.stream_lock));
            let segment = ptr.output_segment;
            glib::ffi::g_rec_mutex_unlock(mut_override(&ptr.stream_lock));
            from_glib_none(&segment as *const gst::ffi::GstSegment)
        }
    }
}

impl<O: IsA<VideoEncoder>> VideoEncoderExtManual for O {}

impl HasStreamLock for VideoEncoder {
    fn stream_lock(&self) -> *mut glib::ffi::GRecMutex {
        let encoder_sys: *const ffi::GstVideoEncoder = self.to_glib_none().0;
        unsafe { mut_override(&(*encoder_sys).stream_lock) }
    }

    fn element_as_ptr(&self) -> *const gst::ffi::GstElement {
        self.as_ptr() as *const gst::ffi::GstElement
    }
}
