/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <cassert>
#include <string>

#include <boost/optional.hpp>
#include <boost/unordered_map.hpp>

#include "libebook_utils.h"
#include "EBOOKMemoryStream.h"
#include "IMPHeader.h"
#include "IMPResourceDir.h"

using boost::optional;
using boost::shared_ptr;

using std::string;

namespace libebook
{

namespace
{

template<class Selector>
class ResourceStream : public WPXInputStream
{
public:
  ResourceStream(shared_ptr<WPXInputStream> strm, shared_ptr<IMPResourceDirImpl> resourceDir);

  virtual bool isOLEStream();
  virtual WPXInputStream *getDocumentOLEStream(const char *name);

  virtual const unsigned char *read(unsigned long numBytes, unsigned long &numBytesRead);
  virtual int seek(long offset, WPX_SEEK_TYPE seekType);
  virtual long tell();
  virtual bool atEOS();

private:
  const shared_ptr<WPXInputStream> m_stream;
  const shared_ptr<IMPResourceDirImpl> m_resourceDir;
};

struct NameSelector
{
  static WPXInputStream *getStream(shared_ptr<IMPResourceDirImpl> resourceDir, const char *name);
};

struct TypeSelector
{
  static WPXInputStream *getStream(shared_ptr<IMPResourceDirImpl> resourceDir, const char *name);
};

}

class IMPResourceDirImpl
{
  // -Weffc++
  IMPResourceDirImpl(const IMPResourceDirImpl &other);
  IMPResourceDirImpl &operator=(const IMPResourceDirImpl &other);

  struct ResourceInfo
  {
    ResourceInfo();

    unsigned offset;
    unsigned length;
    optional<string> type;
  };

  typedef boost::unordered_map<string, ResourceInfo> ResourceMap_t;
  typedef boost::unordered_map<string, ResourceMap_t::const_iterator> TypeMap_t;

public:
  IMPResourceDirImpl(WPXInputStream *input, unsigned files, unsigned version);

  WPXInputStream *getDirStream() const;

  WPXInputStream *getResourceByName(const char *name) const;
  WPXInputStream *getResourceByType(const char *type) const;

private:
  ResourceMap_t::const_iterator findResourceByType(const char *type) const;

  WPXInputStream *createStream(const ResourceInfo &info) const;

private:
  WPXInputStream *m_stream;
  unsigned m_start;
  unsigned m_length;
  mutable ResourceMap_t m_resourceMap;
  mutable TypeMap_t m_typeMap;
};

namespace
{

string readFileType(WPXInputStream *const input)
{
  const unsigned char *const data = readNBytes(input, 4);
  const string fileName(data, data + ((0 == data[3]) ? 3 : 4));
  return fileName;
}

WPXInputStream *NameSelector::getStream(const shared_ptr<IMPResourceDirImpl> resourceDir, const char *const name)
{
  return resourceDir->getResourceByName(name);
}

WPXInputStream *TypeSelector::getStream(const shared_ptr<IMPResourceDirImpl> resourceDir, const char *const name)
{
  return resourceDir->getResourceByType(name);
}

template<class Selector>
ResourceStream<Selector>::ResourceStream(const shared_ptr<WPXInputStream> strm, const shared_ptr<IMPResourceDirImpl> resourceDir)
  : m_stream(strm)
  , m_resourceDir(resourceDir)
{
}

template<class Selector>
bool ResourceStream<Selector>::isOLEStream()
{
  return true;
}

template<class Selector>
WPXInputStream *ResourceStream<Selector>::getDocumentOLEStream(const char *const name)
{
  return Selector::getStream(m_resourceDir, name);
}

template<class Selector>
const unsigned char *ResourceStream<Selector>::read(const unsigned long numBytes, unsigned long &numBytesRead)
{
  return m_stream->read(numBytes, numBytesRead);
}

template<class Selector>
int ResourceStream<Selector>::seek(const long offset, const WPX_SEEK_TYPE seekType)
{
  return m_stream->seek(offset, seekType);
}

template<class Selector>
long ResourceStream<Selector>::tell()
{
  return m_stream->tell();
}

template<class Selector>
bool ResourceStream<Selector>::atEOS()
{
  return m_stream->atEOS();
}

}

IMPResourceDir::IMPResourceDir(WPXInputStream *const input, const IMPHeader &header)
  : m_impl()
{
  input->seek(header.getTOCOffset(), WPX_SEEK_SET);
  m_impl.reset(new IMPResourceDirImpl(input, header.getFileCount(), header.getVersion()));
}

shared_ptr<WPXInputStream> IMPResourceDir::getNameStream() const
{
  const shared_ptr<WPXInputStream> strm(m_impl->getDirStream());
  const shared_ptr<WPXInputStream> resource(new ResourceStream<NameSelector>(strm, m_impl));
  return resource;
}

shared_ptr<WPXInputStream> IMPResourceDir::getTypeStream() const
{
  const shared_ptr<WPXInputStream> strm(m_impl->getDirStream());
  const shared_ptr<WPXInputStream> resource(new ResourceStream<TypeSelector>(strm, m_impl));
  return resource;
}

IMPResourceDirImpl::ResourceInfo::ResourceInfo()
  : offset(0)
  , length(0)
  , type()
{
}

IMPResourceDirImpl::IMPResourceDirImpl(WPXInputStream *const input, const unsigned files, const unsigned version)
  : m_stream(input)
  , m_start(0)
  , m_length(0)
  , m_resourceMap()
  , m_typeMap()
{
  m_start = static_cast<unsigned>(input->tell());
  unsigned headerLength = 0;

  switch (version)
  {
  case 1 :
    headerLength = 10;
    break;
  case 2 :
    headerLength = 20;
    break;
  default :
    EBOOK_DEBUG_MSG(("unknown version %d\n", version));
    throw GenericException();
  }

  const unsigned tocLength = headerLength * files;
  unsigned fileOffset = m_start + tocLength;
  m_length = tocLength;

  for (unsigned i = 0; i != files; ++i)
  {
    const string fileName = readFileType(input);
    ResourceInfo info;

    switch (version)
    {
    case 1 :
      skip(input, 2);
      info.length = readU32(input, true);
      break;
    case 2 :
      skip(input, 4);
      info.length = readU32(input, true);
      info.type = readFileType(input);
      skip(input, 4);
      break;
    default :
      throw GenericException();
    }

    info.length += headerLength;
    info.offset = fileOffset;
    fileOffset += info.length;
    m_length += info.length;

    const ResourceMap_t::const_iterator it = m_resourceMap.insert(ResourceMap_t::value_type(fileName, info)).first;
    if (info.type)
      m_typeMap.insert(TypeMap_t::value_type(get(info.type), it));
  }
}

WPXInputStream *IMPResourceDirImpl::getDirStream() const
{
  m_stream->seek(m_start, WPX_SEEK_SET);
  const unsigned char *const data = readNBytes(m_stream, m_length);
  return new EBOOKMemoryStream(data, m_length);
}

WPXInputStream *IMPResourceDirImpl::getResourceByName(const char *const name) const
{
  WPXInputStream *resource = 0;

  ResourceMap_t::const_iterator it = m_resourceMap.find(name);
  if (m_resourceMap.end() != it)
    resource = createStream(it->second);

  return resource;
}

WPXInputStream *IMPResourceDirImpl::getResourceByType(const char *const type) const
{
  TypeMap_t::const_iterator it = m_typeMap.find(type);
  if (m_typeMap.end() == it)
  {
    const ResourceMap_t::const_iterator resIt = findResourceByType(type);
    it = m_typeMap.insert(TypeMap_t::value_type(type, resIt)).first;
  }
  assert(m_typeMap.end() != it);

  WPXInputStream *resource = 0;
  if (m_resourceMap.end() != it->second)
    resource = createStream(it->second->second);

  return resource;
}

IMPResourceDirImpl::ResourceMap_t::const_iterator IMPResourceDirImpl::findResourceByType(const char *const type) const
{
  ResourceMap_t::iterator it = m_resourceMap.begin();

  for (; m_resourceMap.end() != it; ++it)
  {
    ResourceInfo &info = it->second;
    if (!info.type)
    {
      // sniff the content
      m_stream->seek(info.offset, WPX_SEEK_SET);
      if (1 == readU16(m_stream))
        info.type = readFileType(m_stream);
      else
        info.type = string();
    }

    assert(info.type);

    if (type == get(info.type))
      break;
  }

  return it;
}

WPXInputStream *IMPResourceDirImpl::createStream(const ResourceInfo &info) const
{
  m_stream->seek(info.offset, WPX_SEEK_SET);
  const unsigned char *const data = readNBytes(m_stream, info.length);
  return new EBOOKMemoryStream(data, info.length);
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
