/******************************************************************************
 * This file is part of MPlayer Audio Visualization.                          *
 *                                                                            *
 *                                                                            *
 *  MPlayer Audio Visualization is free software; you can redistribute it     *
 *  and/or modify it under the terms of the GNU General Public License as     *
 *  published by the Free Software Foundation; either version 2 of the        *
 *  License, or (at your option) any later version.                           *
 *                                                                            *
 *  MPlayer Audio Visualization is distributed in the hope that it will be    *
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General *
 *  Public License for more details.                                          *
 *                                                                            * 
 *  You should have received a copy of the GNU General Public License         *
 *  along with MPlayer Audio Visualization; if not, write to the Free         *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA         *
 *  02111-1307  USA                                                           *
 ******************************************************************************/
#ifndef _H_SLAVE_
#define _H_SLAVE_

/* Common area */

#include <config.h>


/* Global Functions */

/**
 * Init slave. It will launch a thread that handle input via stdin.
 *
 * @return 0 on success, -1 otherwise
 */
int slave_init();

/**
 * Stop the slave thread and clean up things.
 */
void slave_quit();


#ifndef _C_SLAVE_


#else // _C_SLAVE_

/*
 * Local Area: use only in slave.c 
 */

#include <SDL.h>
#include <SDL_thread.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <mpav.h>
#include <screen.h>
#include <event.h>
#include <msglist.h>


#define __slave_whitespace( c ) ( ( c == ' ' ) || ( c == '\t' ) )
#define __slave_post_event( e ) {  \
  SDL_Event event; \
  event.type = SDL_KEYDOWN; \
  event.key.keysym.sym = e; \
  SDL_PushEvent( &event );  \
  event.type = SDL_KEYUP;   \
  SDL_PushEvent( &event );  \
}



/* Taken from GNU Readline Library docs */
typedef struct 
{
  char *name;         /**< User printable name of the function. */
  rl_icpfunc_t *func; /**< Function to call to do the job. */
  char *doc;          /**< Documentation for this function.  */
} command_t;

/* Local Vars */
static SDL_Thread *slave_thread = NULL; /**< event thread pointer */


/**
 * Get the word from string, returning a pointer to it. This function takes quotes in account!
 * @warning this trashes string, adding '\0' to it to delimit words!
 *
 * @param string the string you want to extract words
 * @return pointer to the first word.
 */
char *__slave_get_word( char *string );

/**
 * Quit program
 *
 * @param argument discarded
 * @return always 0
 */
int __slave_com_quit __P( ( char *argument ) );

/**
 * Print out help for argument, or for all of the commands if no argument present.
 * 
 * @param argument the command name to display help about or NULL to display help to all commands.
 * @return always 0
 */
int __slave_com_help __P( ( char *argument ) );

/**
 * Set resolution
 * 
 * @param argument <width> <height>
 * @return 0 on success; -1 on argument error; -2 on SDL SetVideoMode error
 */
int __slave_com_sr   __P( ( char *argument ) );

/**
 * Increase resolution based on some pre-defined resolutions
 * 
 * @param argument discarded
 * @return always 0
 */
int __slave_com_ir   __P( ( char *argument ) );

/**
 * Decrease resolution based on some pre-defined resolutions
 * @param argument discarded
 * @return always 0
 */
int __slave_com_dr   __P( ( char *argument ) );

/**
 * Toggle fullscreen
 * 
 * @param argument discarded
 * @return always 0
 */
int __slave_com_fs   __P( ( char *argument ) );

/**
 * Toggle display of messages on screen
 * 
 * @param argument discarded
 * @return always 0
 */
int __slave_com_m    __P( ( char *argument ) );

/**
 * Create message (without shadow)
 * 
 * @param argument <fontfile> <ptsize> <x> <y> <fgcolor> <msg>
 * @return 0 if it was created or -1 if not. The message ID will be printed out to stdout
 */
int __slave_com_cm   __P( ( char *argument ) );

/**
 * Create message with shadow
 * 
 * @param argument <fontfile> <ptsize> <x> <y> <fgcolor> <shx> <shy> <shcolor>  <msg>
 * @return 0 if it was created or -1 if not. The message ID will be printed out to stdout
 */
int __slave_com_cms  __P( ( char *argument ) );

/**
 * List messages
 * 
 * @param argument discarded
 * @return always 0
 */
int __slave_com_lm   __P( ( char *argument ) );


/**
 * Destroy message
 *
 * @param argument the message ID
 * @return 0 if it was destroyed, -1 if not
 */
int __slave_com_dm   __P( ( char *argument ) );

command_t commands[] = {
  { "quit", __slave_com_quit, "Quit" },
  { "exit", __slave_com_quit, "Synonym for 'quit'" },
  { "help", __slave_com_help, "Display this help" },
  { "?",    __slave_com_help, "Synonym for 'help'" },
  { "sr",   __slave_com_sr,   "Set resolution. Usage: sr <width> <height>" },
  { "ir",   __slave_com_ir,   "Increase resolution" },
  { "dr",   __slave_com_dr,   "Decrease resolution" },
  { "fs",   __slave_com_fs,   "Toggle full screen" },
  { "m",    __slave_com_m,    "Toggle message display on screen" },
  { "cm",   __slave_com_cm,   "Create message without shadow and return it's ID. Usage: cm <fontfile> <ptsize> <x> <y> <fgcolor> <msg>" },
  { "cms",  __slave_com_cms,  "Create message with shadow and return it's ID. Usage: cms <fontfile> <ptsize> <x> <y> <fgcolor> <shx> <shy> <shcolor> <msg>" },
  { "lm",   __slave_com_lm,   "List messages" },
  { "dm",   __slave_com_dm,   "Destroy message. Usage: dm <ID>" },
  { (char *)NULL, (rl_icpfunc_t *)NULL, (char *)NULL }
};

/* Local functions */

#define __slave_command_error_noargs( cname )  \
   mpav_fmsg( stderr, "ERROR: you must provide arguments to command '" cname "'. Try: help " cname " to see required arguments.\n" );

#define __slave_command_error_misarg( cname, aname ) \
   mpav_fmsg( stderr, "ERROR: you provide argument '" aname "' to command '" cname "'. Try: help " cname " to see required arguments.\n" );

#define __slave_get_next_word( word ) \
   ( word ) ? ( __slave_get_word( word + 1 + strlen( word ) ) ) : ( NULL )


/**
 * Function to be launched as thread. Handle input via stdin
 */
void __slave_thread_func();

/**
 * Remove leading and trailing white space  
 *
 * @param string text to be stripped
 * @return the striped text.
 */
char *__slave_stripwhite( char *string );

/**
 * Find command
 *
 * @param name the command you want to get the struct
 * @return the command struct
 */
command_t *__slave_find_command( char *name );

/**
 * Execute command line
 * 
 * @param line the comand line
 * @return command return or -1 if it doesn't exist.
 */
int __slave_execute_line( char *line );

/**
 * Tell the GNU Readline library how to complete.  We want to try to
 * complete on command names if this is the first word in the line, or
 * on filenames if not.
 */
void __slave_initialize_readline();

/**
 * Attempt to complete on the contents of TEXT. 
 *
 * @param text the word to complete
 * @param start start point in region of rl_line_buffer that contains the word to complete
 * @param end   end point in the region started by START.
 * @return array of matches or NULL if there aren't any.
 */
char **__slave_completion __P( ( const char *text, int start, int end ) );


/**
 * Generator function for command completion.  
 *
 * @param text the text to complete
 * @param state lets us know whether to start from scratch; without any state (i.e. 0), then we start at the top of the list. 
 * @return 
 */
char *__slave_command_generator __P( ( const char *text, int state ) );




#endif // _C_SLAVE_

#endif // _H_SLAVE_
