/***************************** LICENSE START ***********************************

 Copyright 2021 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef MVQFEATURETYPE_H
#define MVQFEATURETYPE_H

#include <QMap>
#include <QPen>
#include <QBrush>
#include <QPixmap>
#include <QString>
#include <QVector>

class MvRequest;
class WmoWeatherSymbol;

class MvQFeatureGroupItem
{
public:
    MvQFeatureGroupItem(const MvRequest& r);
    MvQFeatureGroupItem(QString, QString, QString);

    QString name() const {return name_;}
    QString label() const {return label_;}
    QString description() const {return desc_;}
    static const QMap<QString, MvQFeatureGroupItem*>& items() {return items_;}

protected:
    QString name_;
    QString label_;
    QString desc_;
    static QMap<QString, MvQFeatureGroupItem*> items_;
};

class MvQFeatureType
{
public:
    MvQFeatureType(bool system) : system_(system) {}
    virtual ~MvQFeatureType() {}

    virtual QString type() const = 0;
    virtual QString name() const = 0;
    virtual QString label() const = 0;
    virtual QString description() const = 0;
    virtual QString group() const = 0;
    virtual QString subGroup() const = 0;
    QPixmap pixmap() const {return pix_;}
    virtual QPixmap pixmap(int w, int h, QColor fgCol = QColor(Qt::black), QPen boxPen=QPen(Qt::NoPen), QBrush boxBrush=QBrush(Qt::NoBrush)) const = 0;

    static QVector<MvQFeatureType*> items();
    static MvQFeatureType* find(QString name);
    static void load();
    static int defaultPixSize() {return defaultPixSize_;}

protected:
    bool system_{true};
    static int defaultPixSize_;
    QPixmap pix_;

private:
    MvQFeatureType& operator = (const MvQFeatureType&);
    static void load_system_feature_items();
    static void load_user_feature_items();
    static void load_wmo_symbols();
    static void config_loader();

    static QVector<MvQFeatureType*> items_;
    static QStringList groups_;
};

class MvQStandardFeatureType : public MvQFeatureType
{
public:
    MvQStandardFeatureType(const MvRequest& r);
    MvQStandardFeatureType(QString, QString, QString, QString, QString, QString, bool = true);

    QString type() const  override {return type_;}
    QString name() const override {return name_;}
    QString label() const override {return label_;}
    QString description() const override;
    QString group() const override;
    QString subGroup() const override;
    QPixmap pixmap(int w, int h, QColor fgCol = QColor(Qt::black), QPen boxPen=QPen(Qt::NoPen), QBrush boxBrush=QBrush(Qt::NoBrush)) const override;
    static QList<MvQFeatureType*> items();

protected:
    static QList<QString> userIconDirs_;
    QString iconPath() const;

    QString type_;
    QString name_;
    QString label_;
    QString desc_;
    QString iconName_;
    QString iconDir_;
    QString group_;
    QString svgTemplateColour_;
};


class MvQWmoSymbolFeatureType : public MvQFeatureType
{
public:
    MvQWmoSymbolFeatureType(WmoWeatherSymbol*);

    QString type() const  override;
    QString name() const override;
    QString label() const override;
    QString description() const override;
    QString group() const override;
    QString subGroup() const override;
    QPixmap pixmap(int w, int h, QColor fgCol = QColor(Qt::black), QPen boxPen=QPen(Qt::NoPen), QBrush boxBrush=QBrush(Qt::NoBrush)) const override;

protected:
    WmoWeatherSymbol* symbol_{nullptr};
};

#endif // MVQFEATURETYPE_H
