#pragma once
#include "SQLite/sqlite3.h"
#include "Core/Io/Url.h"
#include "Core/Array.h"
#include "Core/Timing.h"
#include "SQL.h"

namespace sql {

	/**
	 * Implementation of a connection to SQLite databases.
	 */
	class SQLite : public DBConnection {
		STORM_CLASS;
	public:
		// Create a connection to an SQLite database in a file. The Url has to refer to a file that
		// resides in the filesystem.
		STORM_CTOR SQLite(Url *str);

		// Create a connection to a temporary, in-memory database.
		STORM_CTOR SQLite();

		// Destroy. Calls 'close'.
		virtual ~SQLite();

		// Prepare a statement.
		using DBConnection::prepare;
		Statement *STORM_FN prepare(Str *query) override;

		// Close the connection.
		void STORM_FN close() override;

		// Returns all names of tables in SQLite connection in an Array of strings.
		Array<Str*> *STORM_FN tables() override;

		// Returns a Schema for SQLite connection.
		MAYBE(Schema *) STORM_FN schema(Str *table) override;

		// Migrate.
		void STORM_FN migrate(Migration *migration) override;

		// Get features.
		virtual features::DBFeatures STORM_FN features() const override;

		// Set the timeout when accessing a locked database.
		void STORM_ASSIGN timeout(Duration timeout);

		// Get the timeout.
		Duration STORM_FN timeout() { return dbTimeout; }

	protected:
		// Visitor.
		QueryStr::Visitor *STORM_FN visitor() const override;

		// Transaction management.
		void STORM_FN beginTransaction() override;
		void STORM_FN endTransaction(Transaction::End end) override;

	private:
		// The SQLite3 object.
		UNKNOWN(PTR_NOGC) sqlite3 *db;

		// Timeout set for the database.
		Duration dbTimeout;

		// Query execution functions.
		void query(const char *query);
		void query(Str *query);
		void query(QueryStr *query);

		// Migrate a single table.
		void migrateTable(Migration::Table *migration);

	public:

		/**
		 * Prepared statements in SQLite.
		 */
		class Stmt : public Statement {
			STORM_CLASS;
		public:
			// Create. Called by SQLite class.
			Stmt(SQLite *owner, Str *statement);

			// Destroy.
			virtual ~Stmt();

			// Bind parameters.
			virtual void STORM_FN bind(Nat pos, Str *str) override;
			virtual void STORM_FN bind(Nat pos, Bool b) override;
			virtual void STORM_FN bind(Nat pos, Int i) override;
			virtual void STORM_FN bind(Nat pos, Long i) override;
			virtual void STORM_FN bind(Nat pos, Float f) override;
			virtual void STORM_FN bind(Nat pos, Double d) override;
			virtual void STORM_FN bindNull(Nat pos) override;

			// Execute.
			Statement::Result STORM_FN execute() override;

			// Finalize the statement.
			void STORM_FN finalize() override;

		protected:
			// Dispose results, re-create the statement.
			void STORM_FN disposeResult() override;

			// Get the next row.
			Maybe<Row> STORM_FN nextRow() override;

			// Get the last row id.
			Int STORM_FN lastRowId() override { return lastId; }

			// Get the number of changed rows.
			Nat STORM_FN changes() override { return lastChanges; }

		private:
			// Owner.
			SQLite *owner;

			// Prepared statement.
			UNKNOWN(PTR_NOGC) sqlite3_stmt *stmt;

			// Last row id.
			Int lastId;

			// Last number of changed rows.
			Nat lastChanges;

			// Is the statement in a clean state?
			Bool isClean;

			// Do we have a row of results ready?
			Bool hasRow;

			// Do we have more results to read using step?
			Bool moreRows;

			// Ensure that the statement is ready for manipulation.
			void reset();
		};


		/**
		 * Visitor to transform query strings into proper queries.
		 */
		class Visitor : public QueryStr::Visitor {
			STORM_CLASS;
		public:
			STORM_CTOR Visitor();
			void STORM_FN lastRowId(StrBuf *to) override;
			void STORM_FN type(StrBuf *to, QueryType type) override;
		};
	};

}
