/*
 * twpsk:  A gui application for PSK
 * Copyright (C) 1999-2014 Ted Williams WA0EIR 
 * widgets.C
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * Version: 4.3 - Aug 2017
 */

#include "GUI.h"
#include "twpskWids.h"
#include "twpskWF.h"
#include "icons/sa.xbm"
#include "icons/wf.xbm"

extern Scope scope; 
extern Disp disp;
extern Wids pskwids;
extern XtAppContext ac;
extern Pixel lightBG, darkBG;
extern int use_stereo;
#if (MAKE_ICON == 1) && (HAVE_X11_XPM_H == 1) && (HAVE_LIBXPM == 1)
extern Pixmap pixmap;
#endif

/*
 *  buildWidgets Method
 *  Creates all of the widgets for the interface
 */
void Wids::buildWidgets (Widget shell, AppRes *appRes)
{  
   Dimension d1, d2;

   /*
    * mainForm - main form
    */
   mainForm = XtVaCreateWidget ("mainForm", xmFormWidgetClass, shell,
      XmNverticalSpacing, SPACE_B,
      XmNhorizontalSpacing, SPACE_B,
      NULL);

   /*
    * label1 - callsign label
    */
   label1 = XtVaCreateManagedWidget ("label1", xmLabelWidgetClass, mainForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * menubar -  menubar and submenus
    */
   menuBar = XmCreateMenuBar (mainForm, (char *)"menuBar", NULL, 0);
   /* All bar buttons get the tear off treatment */
   XtSetArg (args[i], XmNtearOffModel, XmTEAR_OFF_ENABLED); i++;

   fileMenu = XmCreatePulldownMenu (menuBar, (char *)"filePulldown", args, i);
   controlsMenu = XmCreatePulldownMenu (menuBar, (char *)"controlsPulldown",
                  args, i);
   seekMenu = XmCreatePulldownMenu (menuBar, (char *)"SeekPulldown", args, i);

   /*
    * File Cascade Button and submenu buttons
    */
   fileCas = XtVaCreateManagedWidget ("File", xmCascadeButtonWidgetClass,
      menuBar,
      XmNmnemonic, 'F',
      XmNsubMenuId, fileMenu,
      NULL);

   /* Create button to match filenames in resource - appRes->buttonNames */
   createButtons (shell, appRes->buttonNames); 

   /*
    * Controls Cascade Button and submenu buttons
    */
   controlsCas = XtVaCreateManagedWidget ("Controls",
      xmCascadeButtonWidgetClass, menuBar,
      XmNmnemonic, 'C',
      XmNsubMenuId, controlsMenu,
      NULL);
   XmStringFree (xs[0]);

   /* Video menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-V");
   videoBtn = XtVaCreateManagedWidget("Video",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>V",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'V',
      NULL);
   XmStringFree (xs[0]);

   /* Separator */
   sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, controlsMenu,
      NULL);

   /* Recv menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-R");
   rxBtn = XtVaCreateManagedWidget("Recv",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>R",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'R',
      NULL);
   XmStringFree (xs[0]);

   /* Xmit menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-X");
   txBtn = XtVaCreateManagedWidget("Xmit",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>X",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'X',
      NULL);
   XmStringFree (xs[0]);

   /* Tune menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-T");
   tuneBtn = XtVaCreateManagedWidget("Tune",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>T",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'T',
      NULL);
   XmStringFree (xs[0]);

   /* Separator */
   sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, controlsMenu,
      NULL);

   /* Broad menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-B");
   broadBtn = XtVaCreateManagedWidget("Broad",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>B",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'B',
      NULL);
   XmStringFree (xs[0]);

   /* Medium menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-M");
   mediumBtn = XtVaCreateManagedWidget("Medium",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>M",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'M',
      NULL);
   XmStringFree (xs[0]);

   /* Narrow menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-N");
   narrowBtn = XtVaCreateManagedWidget("Narrow",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>N",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'N',
      NULL);
   XmStringFree (xs[0]);

   /* Separator */
   sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, controlsMenu,
      NULL);

   /* Spectrum menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-S");
   saBtn = XtVaCreateManagedWidget("Spectrum",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>S",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'S',
      NULL);
   XmStringFree (xs[0]);

   /* Waterfall menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-W");
   wfBtn = XtVaCreateManagedWidget("Waterfall",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>W",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'W',
      NULL);
   XmStringFree (xs[0]);

   /* Waterfall direction menu toggle button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-F");
   wfDirTB = XtVaCreateManagedWidget ("Flow Up",
      xmToggleButtonWidgetClass, controlsMenu,
      XmNvisibleWhenOff, True,
      XmNindicatorType, XmN_OF_MANY,
      XmNaccelerator, "Ctrl<Key>F",
      XmNacceleratorText, xs[0],
//TJW we have two F's?  ... if you count the main menu
      //XmNmnemonic, 'F',
      NULL);
   XmStringFree (xs[0]);

   disp.setWF_up (XmToggleButtonGetState (wfDirTB));

   /* Separator */
   sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, controlsMenu,
      NULL);

   /* Local Caps menu toggle button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-L");
   localCapsTB = XtVaCreateManagedWidget ("Local Caps",
      xmToggleButtonWidgetClass, controlsMenu,
      XmNvisibleWhenOff, True,
      XmNindicatorType, XmN_OF_MANY,
      XmNaccelerator, "Ctrl<Key>L",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'L',
      NULL);
   XmStringFree (xs[0]);

   pskwids.setDoLocalCaps (XmToggleButtonGetState (localCapsTB));

   /* Separator */
   sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, controlsMenu,
      NULL);

   /* Decoder window */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-O");
   openBtn = XtVaCreateManagedWidget("Open Decoder",
      xmPushButtonWidgetClass, controlsMenu,
      XmNaccelerator, "Ctrl<Key>O",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'O',
      NULL);
   XmStringFree (xs[0]);

   /* End of Controls Menu */


   /* 
    * Seek Cascade Button and submenu.  
    */
   seekCas = XtVaCreateManagedWidget ((char *)"Seek",
      xmCascadeButtonWidgetClass, menuBar,
      XmNmnemonic, 'S',
      XmNsubMenuId, seekMenu,
      NULL);

   /* Seek Up menu button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-U");
   upBtn = XtVaCreateManagedWidget("Up",
      xmPushButtonWidgetClass, seekMenu,
      XmNaccelerator, "Ctrl<Key>U",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'U',
      NULL);
   XmStringFree (xs[0]);
   
   /* Seek Down button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-D");
   downBtn = XtVaCreateManagedWidget("Down",
      xmPushButtonWidgetClass, seekMenu,
      XmNaccelerator, "Ctrl<Key>D",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'D',
      NULL);
   XmStringFree (xs[0]);
   
   /* Seek Halt button */
   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-H");
   haltBtn = XtVaCreateManagedWidget("Halt",
      xmPushButtonWidgetClass, seekMenu,
      XmNaccelerator, "Ctrl<Key>H",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'H',
      NULL);
   XmStringFree (xs[0]);

   /* Help Menu Button */
   helpCas = XtVaCreateManagedWidget ("Help", xmCascadeButtonWidgetClass,
      menuBar,
      XmNmnemonic, 'H',
      XmNsubMenuId, helpMenu,
      NULL);

   /*
    * Help Cascade Button and submenu
    */
   i = 0;
   xs[i] = XmStringCreateLocalized ((char *)"About"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Introduction"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Menu Bar"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Help Windows"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Main Window"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Tuning"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Secondary Windows"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Sending Files"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Macros Commands"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Resources"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Accelerator Keys"); i++;
                                                                                
   helpMenu = XmVaCreateSimplePulldownMenu (menuBar, (char *)"helpMenu",
      3, helpCB,
      XmVaPUSHBUTTON, xs[0], 'A', NULL, NULL,
      XmVaPUSHBUTTON, xs[1], 'I', NULL, NULL,
      XmVaPUSHBUTTON, xs[2], 'B', NULL, NULL,
      XmVaPUSHBUTTON, xs[3], 'H', NULL, NULL,
      XmVaPUSHBUTTON, xs[4], 'M', NULL, NULL,
      XmVaPUSHBUTTON, xs[5], 'T', NULL, NULL,
      XmVaPUSHBUTTON, xs[6], 'W', NULL, NULL,
      XmVaPUSHBUTTON, xs[7], 'S', NULL, NULL,
      XmVaPUSHBUTTON, xs[8], 'C', NULL, NULL,
      XmVaPUSHBUTTON, xs[9], 'R', NULL, NULL,
      XmVaPUSHBUTTON, xs[10], 'K', NULL, NULL,
      NULL);
                                                                                
   for (j=0; j<i; j++)
   {
      XmStringFree (xs[j]);
   }

   XtVaSetValues (helpMenu,
      XmNtearOffModel, XmTEAR_OFF_ENABLED,
      NULL);

   /* Main Menu Properties */
   XtVaSetValues (menuBar,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, label1,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      XmNmenuHelpWidget, helpCas,
      NULL);

/* 
 * End of menu stuff
 */

   /*
    * bottomForm - for the three frames across the bottom
    */

   bottomForm = XtVaCreateWidget ("bottomForm", xmFormWidgetClass, mainForm,
      XmNleftOffset, SPACE_B/2,
      XmNhorizontalSpacing, SPACE_B,
      XmNverticalSpacing, SPACE_B,
      XmNtopAttachment, XmATTACH_NONE,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNleftOffset, SPACE_B,
      XmNrightAttachment, XmATTACH_FORM,
      XmNrightOffset, 2 * SPACE_B,
      NULL);

   /*
    * frame for display controls
    */
   f1 = XtVaCreateWidget ("f1", xmFrameWidgetClass, bottomForm,
      XmNshadowThickness, 3,
      XmNshadowType, XmSHADOW_OUT,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   displayForm = XtVaCreateWidget ("displayForm", xmFormWidgetClass, f1,
      XmNhorizontalSpacing, SPACE_B,
      XmNverticalSpacing, SPACE_B,
      NULL);
   
   /*
    * displayRC - row/column for label, displayRB, ,bwRB and scopeDA
    */
   displayRC = XtVaCreateWidget ("displayRC", xmRowColumnWidgetClass,
      displayForm,
      XmNentryAlignment, XmALIGNMENT_CENTER,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * displayLabel - label for display controls
    */
   displayLabel = XtVaCreateManagedWidget ("Display Controls",
      xmLabelWidgetClass, displayRC,
       NULL);

   /*
    * displayRB - spectrum analyser/waterfall radiobox
    */
   i = 0;
   XtSetArg (args[i], XmNorientation, XmHORIZONTAL); i++;
   XtSetArg (args[i], XmNmarginWidth, 0); i++;
   XtSetArg (args[i], XmNspacing, 4), i++;

   displayRB = XmCreateRadioBox (displayRC, (char *)"displayRB", args, i);

   /* Pixmaps for sa and wf ToggleButtons */
   fg = WhitePixelOfScreen (XtScreen (shell));
   bg = BlackPixelOfScreen (XtScreen (shell));

   saPixmap = XCreatePixmapFromBitmapData (XtDisplay (shell),
      RootWindowOfScreen (XtScreen (shell)),
      (char *)sa_bits, sa_width, sa_height, bg, fg,
      DefaultDepthOfScreen (XtScreen (shell)));

   wfPixmap = XCreatePixmapFromBitmapData (XtDisplay (shell),
      RootWindowOfScreen (XtScreen (shell)),
      (char *) wf_bits, wf_width, wf_height, bg, fg,
      DefaultDepthOfScreen (XtScreen (shell)));

   saTB = XtVaCreateManagedWidget ("saTB", xmToggleButtonWidgetClass, displayRB,
      XmNshadowThickness, 3,
      XmNlabelType, XmPIXMAP,
      XmNlabelPixmap, saPixmap,
      NULL);
      
   wfTB = XtVaCreateManagedWidget ("wfTB", xmToggleButtonWidgetClass, displayRB,
      XmNshadowThickness, 3,
      XmNlabelType, XmPIXMAP,
      XmNlabelPixmap, wfPixmap,
      NULL);

   XtVaGetValues (wfTB,
      XmNset, &i,
      NULL);
   disp.changeDisplay (i);

   /*
    * bwRB - band width selection radio box
    */
   i = 0;
   XtSetArg (args[i], XmNorientation, XmHORIZONTAL); i++;
   XtSetArg (args[i], XmNmarginWidth, 0); i++;
   XtSetArg (args[i], XmNspacing, 3 ), i++;

   bwRB = XmCreateRadioBox (displayRC, (char *)"bwRB", args, i);

   broadTB = XtVaCreateManagedWidget ("B", xmToggleButtonWidgetClass, bwRB,
      XmNshadowThickness, 3,
      XmNmnemonic, 'B',
      NULL);

   mediumTB = XtVaCreateManagedWidget ("M", xmToggleButtonWidgetClass, bwRB,
      XmNshadowThickness, 3,
      XmNmnemonic, 'M',
      NULL);

   narrowTB = XtVaCreateManagedWidget ("N", xmToggleButtonWidgetClass, bwRB,
      XmNshadowThickness, 3,
      XmNmnemonic, 'N',
      NULL);

   /*
    * find the toggle button set by the resource file
    * and set fftSample.
    */
   if (XmToggleButtonGetState(broadTB) == 1)
      fftSample = 2;

   if (XmToggleButtonGetState(mediumTB) == 1)
      fftSample = 4;

   if (XmToggleButtonGetState(narrowTB) == 1)
      fftSample = 8;

   /*
    * scopeDA - Scope drawing area 
    */
   scopeDA = XtVaCreateWidget ("scopeDA", xmDrawingAreaWidgetClass,
      displayForm,
      XmNtraversalOn, False,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, displayRC,
      XmNtopWidget, displayRC,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * frame for Rx and Tx controls
    */
   f2 = XtVaCreateWidget ("f2", xmFrameWidgetClass, bottomForm,
      XmNshadowThickness, 3,
      XmNshadowType, XmSHADOW_OUT,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, f1,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * controls form for controlsRC and T/R RB
    */
   controlsForm = XtVaCreateWidget ("controlsForm", xmFormWidgetClass, f2,
      XmNverticalSpacing, SPACE_B,
      XmNhorizontalSpacing, SPACE_B,
      NULL);

   /*
    * controlsRC - row column for rx/tx freq and controls
    */
   controlsRC = XtVaCreateWidget ("controlsRC", xmRowColumnWidgetClass,
      controlsForm,
      XmNentryAlignment, XmALIGNMENT_CENTER,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    *  rxLabel - label for rx controls
    */
   rxLabel = XtVaCreateManagedWidget ("Rx Freq", xmLabelWidgetClass, controlsRC,
      NULL);

   /* Sync class with widget */
   commControl(COMM_RXCH, COMM_AFC, appRes->afc);

   /*
    * rxComboForm - for recv controls - freq, up/down, and AFC
    */      
   genFrame = XtVaCreateManagedWidget ("genFrame",xmFrameWidgetClass,
      controlsRC,
      XmNshadowType, XmSHADOW_ETCHED_IN,
      XmNshadowThickness, 3,
      NULL);

   rxComboForm = XtVaCreateWidget ("rxComboForm", xmFormWidgetClass, genFrame,
      XmNverticalSpacing, SPACE_A,
      XmNhorizontalSpacing, SPACE_A,
      NULL);
  
   /*
    * rxFreqText - Recv Freq Textfield 
    */
   if (appRes->rxFreq == 0.0)
      appRes->rxFreq = 1000.0;

   commControl(COMM_RXCH, COMM_FREQ, (int)(appRes->rxFreq*100));
   sprintf (str, "%4.1f", appRes->rxFreq);

   rxFreqTF = XtVaCreateManagedWidget ("rxFreqText", xmTextFieldWidgetClass,
      rxComboForm,
      XmNuserData, (XtPointer) 'R',
      XmNshadowThickness, 0,
      XmNmaxLength, 6,
      XmNcolumns, 6,
      XmNmarginHeight, 10,
      XmNvalue, str,
      XmNtopAttachment,XmATTACH_FORM,
      XmNbottomAttachment,XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * rxUpArrow - Arrow button for rx freq Up
    */
   rxUpArrow = XtVaCreateManagedWidget ("rxUpArrow", xmArrowButtonWidgetClass,
      rxComboForm,
      XmNuserData, (XtPointer) 'R',
      XmNshadowThickness, 0,
      XmNwidth, ARROW_WIDTH,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_POSITION,
      XmNbottomPosition, 50,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, rxFreqTF,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * rxDownArrow - Arrow button for rx freq Down
    */
   rxDnArrow = XtVaCreateManagedWidget ("rxUpArrow", xmArrowButtonWidgetClass,
      rxComboForm,
      XmNuserData, (XtPointer) 'R',
      XmNshadowThickness, 0,
      XmNwidth, ARROW_WIDTH,
      XmNarrowDirection, XmARROW_DOWN,
      XmNtopAttachment, XmATTACH_POSITION,
      XmNtopPosition, 50,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, rxFreqTF,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * afcTB - Create Toggle button for AFC and set to start up value 
    */
   afcTB = XtVaCreateManagedWidget ("AFC", xmToggleButtonWidgetClass,
      rxComboForm,
      XmNset, appRes->afc,
      XmNalignment, XmALIGNMENT_BEGINNING,
      XmNshadowThickness, 0,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, rxUpArrow,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
      
   /* Sync class with widget */
   commControl(COMM_RXCH, COMM_AFC, appRes->afc);


   /*
    *  txLabel - label for tx controls
    */
   txLabel = XtVaCreateManagedWidget ("Tx Freq", xmLabelWidgetClass,
      controlsRC,
      NULL);

   /*
    * txComboForm - for xmit controls - freq, up/down, and Net
    */      
   genFrame = XtVaCreateManagedWidget ("genFrame",xmFrameWidgetClass,
      controlsRC,
      XmNshadowType, XmSHADOW_ETCHED_IN,
      XmNshadowThickness, 3,
      NULL);

   txComboForm = XtVaCreateWidget ("txComboForm", xmFormWidgetClass, genFrame,
      XmNverticalSpacing, SPACE_A,
      XmNhorizontalSpacing, SPACE_A,
      NULL);   

   /*
    * txFreqText - Xmit freq Text Field
    */
   if (appRes->txFreq == 0.0)
      appRes->txFreq = 1000.0;

   sprintf (str, "%4.1f", appRes->txFreq);

   txFreqTF = XtVaCreateManagedWidget ("txFreqText", xmTextFieldWidgetClass,
      txComboForm,
      XmNshadowThickness, 0,
      XmNuserData, (XtPointer) 'T',
      XmNmaxLength, 6,
      XmNcolumns, 6,
      XmNmarginHeight, 10,
      XmNvalue, str,
      XmNtopAttachment,XmATTACH_FORM,
      XmNbottomAttachment,XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * txUpArrow - Arrow button for tx freq Up
    */
   txUpArrow = XtVaCreateManagedWidget ("txUpArrow", xmArrowButtonWidgetClass,
      txComboForm,
      XmNuserData, (XtPointer) 'T',
      XmNshadowThickness, 0,
      XmNwidth, ARROW_WIDTH,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_POSITION,
      XmNbottomPosition, 50,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, txFreqTF,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * txDownArrow - Arrow button for tx freq Down
    */
   txDnArrow = XtVaCreateManagedWidget ("txUpArrow", xmArrowButtonWidgetClass,
      txComboForm,
      XmNuserData, (XtPointer) 'T',
      XmNshadowThickness, 0,
      XmNwidth, ARROW_WIDTH,
      XmNarrowDirection, XmARROW_DOWN,
      XmNtopAttachment, XmATTACH_POSITION,
      XmNtopPosition, 50,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, txFreqTF,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * netTB - Toggle button for NET
    */
   netTB = XtVaCreateManagedWidget ("NET", xmToggleButtonWidgetClass,
      txComboForm,
      XmNset, appRes->net,
      XmNshadowThickness, 0,
      XmNalignment, XmALIGNMENT_BEGINNING,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, txUpArrow,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
      
   /* Sync class with widget */
   setNetmode (appRes->net);


   /*
    *  trLabel - label for xmit/rec/tune controls
    */

   trLabel = XtVaCreateManagedWidget ("T/R Controls", xmLabelWidgetClass,
      controlsForm,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, controlsRC,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      
      NULL);

   /*
    * trRB - Radio Box for xmit/recv/tune mode controls
    */
   trRB = XmCreateRadioBox (controlsForm, (char *)"trRB", NULL, 0);

   XtVaSetValues (trRB,
      XmNspacing, 10,
      XmNorientation, XmHORIZONTAL,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, trLabel,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * rxTB - Recv toggle button
    * The rx, tx, and tune togglebuttons have the indicator OFF
    * and the shadow thickness = 2, so they look like regular
    * push buttons.
    */
   rxTB = XtVaCreateManagedWidget ("Recv", xmToggleButtonWidgetClass, trRB,
      XmNset, True,
      XmNindicatorOn, False,
      XmNshadowThickness, 3,
      XmNmnemonic, 'R',
      NULL);

   /*
    * txTB - Xmit toggle button
    */
   txTB = XtVaCreateManagedWidget ("Xmit", xmToggleButtonWidgetClass, trRB,
      XmNindicatorOn, False,
      XmNshadowThickness, 3,
      XmNmnemonic, 'X',
      NULL);

   /*
    * tuneTB - Xmit toggle button
    */
   tuneTB = XtVaCreateManagedWidget ("Tune", xmToggleButtonWidgetClass, trRB,
      XmNindicatorOn, False,
      XmNshadowThickness, 3,
      XmNmnemonic, 'T',
      NULL);

   /* 
    * frame for modes
    */
   f3 = XtVaCreateWidget ("f3", xmFrameWidgetClass, bottomForm,
      XmNshadowThickness, 3,
      XmNshadowType, XmSHADOW_OUT,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, f2,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * modesRC - row column for DCD, QPSK, LSB, and CW ID toggle buttons,
    * and IMD display
    */
   modesRC = XtVaCreateWidget ("modesRC", xmRowColumnWidgetClass, f3,
      XmNisAligned, False,
      XmNadjustMargin, False,
      XmNadjustLast, False,
      XmNspacing, 8 * SPACE_A,
      XmNmarginHeight, 6,
      XmNmarginWidth, 6,
      NULL);

   /*
    * modeLabel - label for mode buttons
    */
   modeLabel = XtVaCreateManagedWidget ("Modes", xmLabelWidgetClass, modesRC,
      XmNalignment, XmALIGNMENT_CENTER,
      NULL); 

   /*
    * dcdTB - DCD toggle button
    */
   dcdTB = XtVaCreateManagedWidget ("DCD", xmToggleButtonWidgetClass, modesRC,
      XmNalignment, XmALIGNMENT_END,
      XmNset, 0,
      XmNshadowThickness, 3,
      NULL);
  
   commControl(COMM_RXCH, COMM_DCDLEVEL, 15);   /* 15 seems to work nicely! */

   /*
    * qpsk - enables Qpsk mode when on
    * defaults to off state, unless set in resource file
    */
   qpskTB = XtVaCreateManagedWidget ("QPSK", xmToggleButtonWidgetClass, modesRC,
      XmNalignment, XmALIGNMENT_END,
      XmNshadowThickness, 3,
      NULL);
 
   /*
    * sbTB - enables LSB mode when on
    * defaults to off state, unless set in resource file
    */
   lsbTB = XtVaCreateManagedWidget ("LSB", xmToggleButtonWidgetClass, modesRC,
      XmNalignment, XmALIGNMENT_END,
      XmNshadowThickness, 3,
      NULL);

   /*
    * cwidTB - enables CW ID mode when on
    */
   cwidTB = XtVaCreateManagedWidget ("CW ID", xmToggleButtonWidgetClass,
      modesRC,
      XmNalignment, XmALIGNMENT_END,
      XmNshadowThickness, 3,
      NULL);

   /*
    * imdFrame - frame for IMD display so it looks like a button
    */
   imdFrame = XtVaCreateWidget ("imdFrame", xmFrameWidgetClass, modesRC,
      XmNshadowThickness, 3,
      XmNshadowType, XmSHADOW_OUT,
      NULL);
      
   /*
    * imdForm - form for IMD display
    */
   imdForm = XtVaCreateWidget ("imdForm", xmFormWidgetClass, imdFrame,
      NULL);
      
   /*
    * imdTF - textfield for IMD display
    */
   imdTF = XtVaCreateManagedWidget ("imdTF", xmTextFieldWidgetClass, imdForm,
      XmNmarginHeight, 2, /* this makes IMD box the same size as the others */
      XmNcolumns, 3,
      XmNshadowThickness, 0,
      XmNvalue, "---",
      XmNcursorPositionVisible, False,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   /*
    * imdLabel - label for IMD display
    */
   imdLabel = XtVaCreateManagedWidget ("IMD", xmLabelWidgetClass, imdForm,
      XmNshadowThickness, 0,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, imdTF,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * Sync classes with lsb and qpsk widgets. They are not set in
    * resource file, but just in case. 
    */
   XtVaGetValues(lsbTB,
      XmNset, &lsb,
      NULL);

   XtVaGetValues(qpskTB,
      XmNset, &qpsk,
      NULL);

   commControl(COMM_TXCH, COMM_QPSK, qpsk);
   commControl(COMM_RXCH, COMM_QPSK, qpsk);
   commControl(COMM_TXCH, COMM_LSB, lsb);
   commControl(COMM_RXCH, COMM_LSB, lsb);


   f4 = XtVaCreateWidget ("f4", xmFrameWidgetClass, mainForm,
      XmNshadowThickness, 5,
      XmNshadowType, XmSHADOW_IN,
      XmNtopAttachment, XmATTACH_NONE,
      XmNbottomAttachment, XmATTACH_WIDGET,
      XmNbottomWidget, bottomForm,
      XmNleftAttachment, XmATTACH_FORM,
      NULL);

   /*
    * wfDA - water fall drawing area
    */
   wfDA = XtVaCreateWidget ("wfDA", xmDrawingAreaWidgetClass, f4,
      XmNheight, WF_HEIGHT,
      XmNwidth, WF_WIDTH,
      XmNbackground, BlackPixelOfScreen (XtScreen (shell)),
      XmNtraversalOn, False,
      NULL);

   /* 
    * paneWin - paned window for rx and tx text wids.`
    */
   paneWin = XtVaCreateWidget ( "paneWin", xmPanedWindowWidgetClass, mainForm,
      XmNbottomOffset, 2 * SPACE_B,      
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, menuBar,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_WIDGET,
      XmNbottomWidget, wfDA,
      NULL);

   /*
    * rxTextSW - Receive Text Scrolled Window
    */
   rxTextSW = XtVaCreateWidget ("rxTextSW", xmScrolledWindowWidgetClass,
   paneWin,
   NULL);

   /* rxText - text widget for received data */
   rxText = XtVaCreateManagedWidget ("rxText", xmTextWidgetClass, rxTextSW,
      XmNeditMode, XmMULTI_LINE_EDIT,
      XmNshadowThickness, 3,
      XmNhighlightThickness, 2,
      XmNeditable, False,
      XmNwordWrap, True,
      XmNcursorPositionVisible, False,
      XmNautoShowCursorPosition, False,
      XmNscrollHorizontal, False,
      XmNuserData, (XtPointer)this,
      NULL);

   /* 
    * txTextSW - Transmit Text Scrolled Window
    */
   txTextSW = XtVaCreateWidget ("txTextSW", xmScrolledWindowWidgetClass,
   paneWin,
   NULL);
   
   /* txText widget - text widget for transmit data */
   txText = XtVaCreateManagedWidget ("txText", xmTextWidgetClass, txTextSW,
      XmNeditMode, XmMULTI_LINE_EDIT,
      XmNeditable, True,
      XmNwordWrap, True,
      XmNshadowThickness, 3,
      XmNhighlightThickness, 2,
      XmNblinkRate, 0,
      XmNscrollHorizontal, False,
      NULL);

   /*
    * Set up rxScrollFlag for normal and scroll mode 
    */
   rxScrollFlag = 0;                         /* rxText scrolling off */

   XtVaGetValues (rxTextSW,                  /* get scrollbar widget */
      XmNverticalScrollBar, &rxScrollBar,    /* for adding to callbacks */
      NULL);
   /*
    * End of main window
    */

   /*
    * videoDiag - create the display dialog/form and its children
    */
   videoDiag = XmCreateFormDialog (shell, (char *)"videoDiag", NULL, 0);
   XtVaSetValues (videoDiag,
      XmNverticalSpacing, SPACE_B,
      XmNhorizontalSpacing, SPACE_B,
      NULL);

   XtVaSetValues (XtParent (videoDiag),   
      XmNtitle, "TWPSK VIDEO",
      #if (MAKE_ICON == 1) && (HAVE_X11_XPM_H == 1) && (HAVE_LIBXPM == 1)
      XmNiconPixmap, pixmap,
      #endif
      XmNmwmDecorations, MWM_DECOR_ALL | MWM_DECOR_RESIZEH,
      NULL);

   /*
    * wfControlsLabel - label for Brightness control
    */
   videoLabel = XtVaCreateManagedWidget ("Video Controls",
      xmLabelWidgetClass,
      videoDiag,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * videoFrame - frame for video controls
    */
   videoFrame = XtVaCreateManagedWidget ("videoFrame", xmFrameWidgetClass,
      videoDiag,
      XmNshadowThickness, 3,
      XmNshadowType, XmSHADOW_OUT,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, videoLabel,
      XmNbottomWidget, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      NULL);

   videoForm = XtVaCreateManagedWidget ("videoForm", xmFormWidgetClass,
      videoFrame,
      XmNverticalSpacing, SPACE_B,
      XmNhorizontalSpacing, SPACE_B,
      NULL);
      
   /*
    * brightLabel - label for Brightness control
    */
   brightLabel = XtVaCreateManagedWidget ("Brightness", xmLabelWidgetClass,
      videoForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * brightScale - fft display brightness
    */
    brightScale = XtVaCreateManagedWidget ("brightScale", xmScaleWidgetClass,
      videoForm,
      XmNorientation, XmHORIZONTAL,
      XmNshowValue, True,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, brightLabel,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

      XtVaGetValues (brightScale,
         XmNvalue, &brightness,
         NULL);
      disp.brightness = brightness;     

   /*
    * fftLabel - label for fft control
    */
   fftLabel = XtVaCreateManagedWidget ("Display Speed", xmLabelWidgetClass,
      videoForm,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, brightScale,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
            
   /*
    * fftScale - sets speed of display
    */
   fftScale = XtVaCreateManagedWidget ("fftSpeedScale", xmScaleWidgetClass,
      videoForm,
      XmNorientation, XmHORIZONTAL,
      XmNshowValue, True,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, fftLabel,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   /*
    * Get the value set by the resource file for the fft setup
    */
   XtVaGetValues (fftScale,
      XmNvalue, &fftSpeed,
      NULL);

   /*
    * displayCloseBtn - close button 
    */  
   displayCloseBtn = XtVaCreateManagedWidget ("Close", xmPushButtonWidgetClass,
      videoDiag,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopOffset, SPACE_B,
      XmNtopWidget, fftScale,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNbottomOffset, SPACE_B,
      XmNleftAttachment, XmATTACH_FORM,
      XmNleftOffset, 5 * SPACE_B,
      XmNrightAttachment, XmATTACH_FORM,
      XmNrightOffset, 5 * SPACE_B,
      NULL);

   /*
    * Create two popup menus for clearing rx and tx text widgets
    */
   i = 0;
   xs[i] = XmStringCreateLocalized ((char *)"Rx Text"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Tx Text"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Clear"); i++;
   xs[i] = XmStringCreateLocalized ((char *)"Ctrl-C"); i++;

   clrRxPopup = XmVaCreateSimplePopupMenu(rxTextSW, (char *)"clrRx", clrTextCB,
      XmVaTITLE, xs[0],
      XmVaSEPARATOR,
      XmVaPUSHBUTTON, xs[2], 'C', (char *)"Ctrl<Key>c", xs[3],
      NULL);

   XtVaSetValues (clrRxPopup,         /* userdata is text wid to clear */
      XmNuserData, (XtPointer)rxText,
      NULL);

   clrTxPopup = XmVaCreateSimplePopupMenu(txTextSW, (char*)"clrTx", clrTextCB,
      XmVaTITLE, xs[1],
      XmVaSEPARATOR,
      XmVaPUSHBUTTON, xs[2], 'C', (char *)"Ctrl<Key>c", xs[3],
      NULL);

/*
 * patch for mouse grab bug in xorg that grabed the
 * clear tx popup window and wouldn't let go - fixed now
 */

#if 0
#if XmVersion >= 2000
   fprintf (stderr, "newer XmVersion %d\n", XmVersion);
   XtVaSetValues (clrTxPopup,
      XmNpopupEnabled, XmPOPUP_DISABLED,
      NULL);
   XtUngrabButton (txTextSW, AnyButton, AnyModifier);
#else
   fprintf (stderr, "older XmVersion %d\n", XmVersion);
   XtVaSetValues (clrTxPopup,
      XmNpopupEnabled, False,
      NULL);
#endif
#endif

   XtVaSetValues (clrTxPopup,         /* userdata is text wid to clear */
      XmNuserData, txText,
      NULL);

   for (j=0; j<i; j++)
   {
      XmStringFree (xs[j]);
   }

   /*      
    * Add Callbacks
    *
    * Controls menu buttons callbacks
    */
   XtAddCallback (rxBtn, XmNactivateCallback, controlsBtnCB, (XtPointer) rxTB);
   XtAddCallback (txBtn, XmNactivateCallback, controlsBtnCB, (XtPointer) txTB);
   XtAddCallback (tuneBtn, XmNactivateCallback, controlsBtnCB,
      (XtPointer) tuneTB);

   XtAddCallback (broadBtn, XmNactivateCallback, controlsBtnCB,
      (XtPointer) broadTB);
   XtAddCallback (mediumBtn, XmNactivateCallback, controlsBtnCB,
      (XtPointer) mediumTB);
   XtAddCallback (narrowBtn, XmNactivateCallback, controlsBtnCB,
      (XtPointer) narrowTB);

   XtAddCallback (saBtn, XmNactivateCallback, controlsBtnCB, (XtPointer) saTB);
   XtAddCallback (wfBtn, XmNactivateCallback, controlsBtnCB, (XtPointer) wfTB);
   XtAddCallback (wfDirTB, XmNvalueChangedCallback, wfDirCB, (XtPointer) NULL);
   XtAddCallback (localCapsTB, XmNvalueChangedCallback, localCapsCB,
                 (XtPointer) NULL);

   /*
    * Setup menu button callbacks -  pop up/pop down the video dialog
    */
   XtAddCallback (videoBtn, XmNactivateCallback, popupDiagCB,
      (XtPointer)videoDiag);
   XtAddCallback (displayCloseBtn, XmNactivateCallback, closeVideoDiagCB,
      (XtPointer) videoDiag);

   /*
    *  SA/WF radio box buttons
    */
   XtAddCallback (saTB, XmNvalueChangedCallback, changeDisplayCB,
      (XtPointer) 0);
   XtAddCallback (wfTB, XmNvalueChangedCallback, changeDisplayCB,
      (XtPointer) 1);

   /* HACK HERE - A radio box does not allow for accelerator keys, so
    * I added them to buttons in the menu bar.  The menu bar CB calls
    * a function that sets the radiobox button, and the valuechanged
    * CB does the rest.  However, the radiobox can get confused when
    * mouse and acc key are intermixed.  These disarm CBs seem to fix it
    */ 
   XtAddCallback (saTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) saTB);
   XtAddCallback (wfTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) wfTB);

   /*
    * Bandwidth radio box buttons
    */
   XtAddCallback (broadTB,  XmNvalueChangedCallback, bwCB, (XtPointer) 2);
   XtAddCallback (mediumTB, XmNvalueChangedCallback, bwCB, (XtPointer) 4);
   XtAddCallback (narrowTB, XmNvalueChangedCallback, bwCB, (XtPointer) 8);

   /* see hack above */
   XtAddCallback (broadTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) broadTB);
   XtAddCallback (mediumTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) mediumTB);
   XtAddCallback (narrowTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) narrowTB);

   /*
    * T/R Menu buttons and Radio box buttons
    */
   XtAddCallback (rxTB, XmNvalueChangedCallback, rxCB, (XtPointer) 0);
   XtAddCallback (txTB, XmNvalueChangedCallback, txCB, (XtPointer) 0);
   XtAddCallback (tuneTB, XmNvalueChangedCallback, tuneCB, (XtPointer) 0);

   /* see hack above */
   XtAddCallback (rxTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) rxTB);
   XtAddCallback (txTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) txTB);
   XtAddCallback (tuneTB, XmNdisarmCallback, controlsBtnCB,
      (XtPointer) tuneTB);

   /*
    * Open Secondary Decoder CB
    */
   XtAddCallback (openBtn, XmNactivateCallback, openBtnCB, NULL);

   /*
    * Rx and Tx freq arrows
    */
   XtAddCallback (rxUpArrow, XmNactivateCallback, arrowCB,
      (XtPointer) rxFreqTF);
   XtAddCallback (rxDnArrow, XmNactivateCallback, arrowCB,
      (XtPointer) rxFreqTF);
   XtAddCallback (txUpArrow, XmNactivateCallback, arrowCB,
      (XtPointer) txFreqTF);
   XtAddCallback (txDnArrow, XmNactivateCallback, arrowCB,
      (XtPointer) txFreqTF);
  
   /*
    * Rx and Tx freqTF
    */
   XtAddCallback (rxFreqTF, XmNfocusCallback, rxFreqFocusCB,
      (XtPointer)&pskwids);
   XtAddCallback (rxFreqTF, XmNlosingFocusCallback, rxFreqFocusCB,
      (XtPointer)&pskwids);
   XtAddCallback (rxFreqTF, XmNmodifyVerifyCallback, freqTextCB,
      (XtPointer) NULL);
   XtAddCallback (rxFreqTF, XmNactivateCallback, freqTextCB,
      (XtPointer) NULL);
   XtAddCallback (txFreqTF, XmNmodifyVerifyCallback, freqTextCB,
      (XtPointer) NULL);
   XtAddCallback (txFreqTF, XmNactivateCallback, freqTextCB,
      (XtPointer) NULL);
   
   /*
    * mode toggle buttons
    */
   XtAddCallback (afcTB, XmNvalueChangedCallback, afcCB, (XtPointer) NULL);
   XtAddCallback (netTB, XmNvalueChangedCallback, netCB, (XtPointer) NULL);
   XtAddCallback (qpskTB, XmNvalueChangedCallback, qpskCB, (XtPointer) NULL);
   XtAddCallback (lsbTB, XmNvalueChangedCallback, lsbCB, (XtPointer) NULL);
   XtAddCallback (dcdTB, XmNvalueChangedCallback, dcdCB, (XtPointer) NULL);


   /*
    * brightness control
    */
   XtAddCallback (brightScale, XmNvalueChangedCallback, brightCB,
      (XtPointer)NULL);
   XtAddCallback (brightScale, XmNdragCallback, brightCB, (XtPointer)NULL);

   /*
    * FFT speed scale
    */
   XtAddCallback (fftScale, XmNvalueChangedCallback, fftScaleCB,
      (XtPointer)NULL);
   XtAddCallback (fftScale, XmNdragCallback, fftScaleCB, (XtPointer)NULL );

   /*
    * wfDA Input callback - for click to tune
    */
   XtAddCallback (wfDA, XmNinputCallback, wfCB, (XtPointer) NULL);

   /*
    * txText Modify/Verify callback - to send text
    */
   XtAddCallback (txText, XmNmodifyVerifyCallback, txTextCB, (XtPointer) 0);
   XtAddCallback (txText, XmNmotionVerifyCallback, txTextCB, (XtPointer) 0);

   /*
    * rxText Scrollbar callbacks
    */
   XtAddCallback (rxScrollBar, XmNdragCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNincrementCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNdecrementCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNpageIncrementCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNpageDecrementCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNtoTopCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNtoBottomCallback, rxScrollBarCB,
      (XtPointer) this);
   XtAddCallback (rxScrollBar, XmNvalueChangedCallback, rxScrollBarCB,
      (XtPointer) this);

   /*
    * expose callback for scope
    */
   XtAddCallback (scopeDA, XmNexposeCallback, scopeCB, (XtPointer) NULL);

   /*
    * seek callback 
    */
   XtAddCallback (upBtn, XmNactivateCallback, seekCB, (XtPointer) 'U');
   XtAddCallback (downBtn, XmNactivateCallback, seekCB, (XtPointer) 'D');
   XtAddCallback (haltBtn, XmNactivateCallback, haltCB, (XtPointer) NULL);

   /*
    * Event handler for clearing rx and tx windows
    */
   XtAddEventHandler (rxText, ButtonPressMask, False, popupHandler,
      (XtPointer) clrRxPopup);
   XtAddEventHandler (txText, ButtonPressMask, False, popupHandler,
      (XtPointer) clrTxPopup);
   XtAddEventHandler (rxTextSW, ButtonPressMask, False, popupHandler,
      (XtPointer) clrRxPopup);
   XtAddEventHandler (txTextSW, ButtonPressMask, False, popupHandler,
      (XtPointer) clrTxPopup);


   /*
    * Add actions for txText widget - sendOver, SendCall, and
    * user defined Macros 
    */
   XtActionsRec actions[] =
   {
      {(char*)"sendOver", sendOver},
      {(char*)"sendHisCall", sendHisCall},
      {(char*)"sendMacro", sendMacro}
   };

   XtAppAddActions (ac, actions, XtNumber(actions));
 
 /*
  * Manage the widgets
  */

   XtManageChild (menuBar);

   XtManageChild (bwRB);
   XtManageChild (displayRB);
   XtManageChild (displayRC);
   XtManageChild (scopeDA);
   XtManageChild (displayForm);
   XtManageChild (f1);

   XtManageChild (txComboForm);
   XtManageChild (rxComboForm);
   XtManageChild (trRB);
   XtManageChild (controlsRC);
   XtManageChild (controlsForm);
   XtManageChild (f2);

   XtManageChild (imdForm);
   XtManageChild (imdFrame);
   XtManageChild (modesRC);
   XtManageChild (f3);


   XtManageChild (bottomForm);

   XtManageChild (wfDA);
   XtManageChild (txTextSW);
   XtManageChild (rxTextSW);
   XtManageChild (paneWin);
   XtManageChild (f4);

   XtManageChild (mainForm);

   XtRealizeWidget(shell);

   /* calc the offset for the wfDA frame */
   XtVaGetValues (mainForm,
      XmNwidth, &d1,
      NULL);
   
   XtVaGetValues (f4,
      XmNwidth, &d2,
      NULL);

   XtVaSetValues (f4,
      XmNleftOffset, (d1 - d2) / 2,
      NULL);

   /* Misc. widget stuff */
   XtVaGetValues (rxScrollBar,
      XmNtroughColor, &darkBG,
      XmNbackground, &lightBG,
      NULL);

   XtVaSetValues (rxTB,
      XmNbackground, darkBG,
      NULL);
      
   rxtTB[0] = rxTB;
   rxtTB[1] = txTB;
   rxtTB[2] = tuneTB;
}


/*
 * createButtons - creates buttons and their callbacks
 * for file cascade from the buttonNames resource. Then
 * creates a seperator, the QRT button, and all the callbacks.
 */

void Wids::createButtons (Widget shell, char **buttonNames)
{
   int cnt = 0;
   char file_acc_text[12] = "Ctrl-";
   char file_acc_num[2];
   char file_acc_key[12];
   Widget fileBtns, sep, QRT;

   /*
    * Count the button names for the File pulldown
    */
   while (buttonNames[cnt] != NULL)
      cnt++;

   if (cnt > MAX_BUTTONS)
   {
      /*
       * There are too many button names in Twpsk, so realize
       * the shell and popup an error dialog
       */
       fprintf (stderr,"Too many file names defined in Twpsk\n");
       exit (0);
   }
   /*
    * Create the buttons for the File pulldown
    */   
   for (i=0; i<cnt; i++)
   {
      /* for accelerator text */
      strcpy (file_acc_text, "Ctrl-");
      sprintf (file_acc_num, "%d", i+1);
      strcat (file_acc_text, file_acc_num);
      xs[0] = XmStringCreateLocalized (file_acc_text);

      /* for accelerator key */
      strcpy (file_acc_key, "Ctrl<Key>");
      strcat (file_acc_key, file_acc_num);

      fileBtns = XtVaCreateManagedWidget(buttonNames[i],
         xmPushButtonWidgetClass, fileMenu,
         XmNaccelerator, file_acc_key,
         XmNacceleratorText, xs[0],
         XmNuserData, (XtPointer) buttonNames[i], /*button name */
         NULL);

      XmStringFree (xs[0]);

      XtAddCallback (fileBtns, XmNactivateCallback, fileCB, (XtPointer) shell);
   }

   /*
    * Now create a seperator and the QRT button, and its callback
    */
   sep = XtVaCreateManagedWidget ("sep", xmSeparatorWidgetClass,
      fileMenu,
      NULL);
   sep = sep;              /* TJW silence the compiler error */

   xs[0] = XmStringCreateLocalized ((char *)"Ctrl-Q");
   QRT = XtVaCreateManagedWidget("QRT",xmPushButtonWidgetClass,fileMenu,
      XmNaccelerator, "Ctrl<Key>Q",
      XmNacceleratorText, xs[0],
      XmNmnemonic, 'Q',
      XmNmnemonicCharSet, XmFONTLIST_DEFAULT_TAG,
      NULL);

   XmStringFree (xs[0]);
   XtAddCallback (QRT, XmNactivateCallback, quitCB, (XtPointer) 0);
}
